import subprocess
import sys
import nltk
from nltk.corpus import wordnet

# List of required packages
REQUIRED_PACKAGES = ["nltk"]

def install_packages():
    """Install missing required packages."""
    for package in REQUIRED_PACKAGES:
        try:
            __import__(package)
        except ImportError:
            print(f"Package '{package}' is missing. Attempting to install...")
            try:
                subprocess.check_call([sys.executable, "-m", "pip", "install", package])
            except Exception as e:
                print(f"Failed to install package '{package}': {e}")

# Ensure required packages are installed
install_packages()

# Download required NLTK resources (run once)
nltk.download('wordnet')
nltk.download('averaged_perceptron_tagger')
nltk.download('averaged_perceptron_tagger_eng')

def parse_word_pos(tagged_words, set_stop_word=None, set_negation_word=None, UNK='x', return_string=False, join_negation=False):
    """
    Parses a list of (WORD, POSTAG) tuples and checks if the WORD exists in WordNet.
    """
    if set_stop_word is None:
        set_stop_word = set()
    if set_negation_word is None:
        set_negation_word = set()

    pos_map = {
        'NN': wordnet.NOUN, 'NNS': wordnet.NOUN,
        'VB': wordnet.VERB, 'VBD': wordnet.VERB, 'VBG': wordnet.VERB,
        'VBN': wordnet.VERB, 'VBP': wordnet.VERB, 'VBZ': wordnet.VERB,
        'JJ': wordnet.ADJ, 'JJR': wordnet.ADJ, 'JJS': wordnet.ADJ,
        'RB': wordnet.ADV, 'RBR': wordnet.ADV, 'RBS': wordnet.ADV
    }

    parsed_words = []
    i = 0
    while i < len(tagged_words):
        word, pos = tagged_words[i]

        if word in set_negation_word:
            if join_negation and (i + 1) < len(tagged_words):
                next_word, next_pos = tagged_words[i + 1]
                wn_pos = pos_map.get(next_pos)
                if wn_pos and wordnet.synsets(next_word, pos=wn_pos):
                    parsed_words.append((f"{word}_{next_word}", next_pos))
                    i += 2
                    continue
            parsed_words.append((word, pos))
            i += 1
            continue

        if word in set_stop_word:
            parsed_words.append((UNK, pos))
            i += 1
            continue

        wn_pos = pos_map.get(pos)
        if wn_pos and wordnet.synsets(word, pos=wn_pos):
            parsed_words.append((word, pos))
        else:
            parsed_words.append((UNK, pos))
        
        i += 1

    return " ".join([word for word, _ in parsed_words]) if return_string else parsed_words
